/* $Id: headtracker.c,v 1.12 2009-06-30 19:01:02 anders Exp $ */
/*
 *=============================================================================
 *  headtracker.c - An ARToolKit based head tracker.
 *
 *  Copyright (C) 2006 - 2009  Anders Gidenstam
 *  http://www.gidenstam.org
 *  anders(at)gidenstam.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *=============================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <pthread.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>

//#include <AR/gsub.h>
#include <AR/video.h>
#include <AR/param.h>
#include <AR/ar.h>
//#include <AR/matrix.h>

/******************************************************************************
 * Global data
 */

/* Camera data. */
char*           cparam_name    = "camera_para.dat";
ARParam         cparam;

/* Pattern data. */
char*           patt_name      = "patt.hiro";
int             patt_id;

int             count = 0;
int             thresh = 100;

char*           host = "127.0.0.1";
int             port = 0;
int             period = 100; // [ms]
double          freshness = 0.70;

/* Threads */
pthread_t       trackerThread;
pthread_t       transmitterThread;

/* Shared data */
pthread_mutex_t mutex =  PTHREAD_MUTEX_INITIALIZER;
float           fps   = 0.0;
double          current_patt_trans[3][4]; /* Patt -> Camers CS. */
double          origin_trans_inv[3][4];   /* Camera CS -> User CS. */
double          current_pos[3];           /* x y z of pat in user CS. */
double          current_rot_deg[3];       /* Heading, roll, pitch of pat
                                             in user CS. */
int             startup = 1;

static void   usage(int, char**);
static void   init(int, char**);
static void   cleanup(void);
static void*  tracker(void*);
static void*  transmitter(void*);
static void   encodeFloat(char* buf, float val);
static float  decodeFloat(char* buf);
static void   encodeFixed(char* buf, float val);
static float  decodeFixed(char* buf);
/* Encodes the float as a 16.16 bit fixed-point number. */

/*---------------------------------------------------------------------------*/
int main(int argc, char** argv)
{
  init(argc, argv);

  printf("Options:\n"
	 "  0   exit;\n"
	 "  1   reset origin;\n"
	 "  D/d increase/decrease detection threshold;\n" 
	 "  X/x increase/decrease freshness/smoothing factor;\n" 
	 "  any key - print state\n");
  for (;;) {

    switch (getchar()) {
    case '0':
    case EOF:
      cleanup();
      exit(0);
      break;

    case '1':
      pthread_mutex_lock(&mutex);
      if (arUtilMatInv(current_patt_trans, origin_trans_inv) < 0) {
        printf("headtracker.c: arUtilMatInv() failed.\n");
      }
      pthread_mutex_unlock(&mutex);
      break;

    case 'D':
      pthread_mutex_lock(&mutex);
      thresh += 5;
      if (thresh > 100) {
	thresh = 100;
      }
      pthread_mutex_unlock(&mutex);
      printf("threshold = %d\n", thresh);
      break;

    case 'd':
      pthread_mutex_lock(&mutex);
      thresh -= 5;
      if (thresh < 0) {
	thresh = 0;
      }
      pthread_mutex_unlock(&mutex);
      printf("threshold = %d\n", thresh);
      break;

    case 'X':
      pthread_mutex_lock(&mutex);
      freshness += 0.05;
      if (freshness > 1.00) {
	freshness = 1.00;
      }
      pthread_mutex_unlock(&mutex);
      printf("freshness = %f\n", freshness);
      break;

    case 'x':
      pthread_mutex_lock(&mutex);
      freshness -= 0.05;
      if (freshness < 0.00) {
	freshness = 0.00;
      }
      pthread_mutex_unlock(&mutex);
      printf("freshness = %f\n", freshness);
      break;

    default:
      pthread_mutex_lock(&mutex);
      printf("*** %f (frame/sec)\n", fps);
      printf("Marker at\n"
	     "  pos = [%f %f %f]\n"
	     "  rot: (yaw,roll,pitch) = (%f %f %f)\n",
	     current_pos[0],
	     current_pos[1],
	     current_pos[2],
	     current_rot_deg[0],
	     current_rot_deg[1],
	     current_rot_deg[2]);
      pthread_mutex_unlock(&mutex);
    }
  }
}

/*---------------------------------------------------------------------------*/
static void   usage(int argc, char** argv)
{
  printf("Usage: %s [options] [ARVideo options]\n", argv[0]);
  printf("\nOPTIONS:\n");
  printf(" --help\n"
	 "    Display this message.\n");
  printf(" --host destIP\n"
	 "    Specify UPD destination IP.\n");
  printf(" --port port\n"
	 "    Specify UDP destination port.\n");
  printf(" --period period\n"
	 "    Specify UDP transmission period [milliseconds].\n");
  printf("\n");
  arVideoDispOption();
}

/*---------------------------------------------------------------------------*/
static void   init(int argc, char** argv)
{
  ARParam  wparam;
  char     line[512];
  int      i;
  int      xsize, ysize;

  /* Clear important state variables. */
  current_pos[0] = 0.0;
  current_pos[1] = 0.0;
  current_pos[2] = 0.0;

  current_rot_deg[0] = 0.0;
  current_rot_deg[1] = 0.0;
  current_rot_deg[2] = 0.0;

  bzero(origin_trans_inv, sizeof(origin_trans_inv));

  /* Parse command line */
  line[0] = 0;
  for (i = 1; i < argc; i++) {
    if ((0 == strcmp(argv[i], "--help")) ||
	(0 == strcmp(argv[i], "-h"))) {
      usage(argc, argv);
      exit(0);
    } else if (0 == strcmp(argv[i], "--host") && (++i < argc)) {
      host = argv[i];
    } else if (0 == strcmp(argv[i], "--port") && (++i < argc)) {
      port = strtol(argv[i], NULL, 10);
    } else if (0 == strcmp(argv[i], "--period") && (++i < argc)) {
      period = strtol(argv[i], NULL, 10);
    } else if (i < argc) {
      strcat(line, " ");
      strcat(line, argv[i]);
    } else {
      /* Bad command line. */
      usage(argc, argv);
      exit(-1);
    }
  }

  /* Open the video path */
  if (arVideoOpen(line) < 0) {
    exit(-1);
  }
  /* Find the size of the window */
  if (arVideoInqSize(&xsize, &ysize) < 0) {
    exit(-1);
  }
  printf("Image size: (%d,%d)\n", xsize, ysize);

  /* Set the initial camera parameters */
  if (arParamLoad(cparam_name, 1, &wparam) < 0) {
    printf("headtracker.c: Camera parameter load error !!\n");
    exit(-1);
  }
  arParamChangeSize(&wparam, xsize, ysize, &cparam);
  arInitCparam(&cparam);
  printf("headtracker.c: *** Camera Parameter ***\n");
  arParamDisp(&cparam);
  
  if ((patt_id=arLoadPatt(patt_name)) < 0) {
    printf("headtracker.c: pattern load error !!\n");
    exit(-1);
  }

  if (0 != pthread_create(&trackerThread, NULL, tracker, NULL)) {
    perror("headtracker.c: pthread_create()");
    exit(-1);
  }

  if (port != 0) {
    if (0 != pthread_create(&transmitterThread, NULL, transmitter, NULL)) {
      perror("headtracker.c: pthread_create()");
    }
  }
}

/*---------------------------------------------------------------------------*/
static void cleanup(void)
{
  arVideoCapStop();
  arVideoClose();
  pthread_mutex_destroy(&mutex);
}

/*---------------------------------------------------------------------------*/
static void*  tracker(void* arg)
{
  ARUint8         *dataPtr;
  ARMarkerInfo    *marker_info;
  int             marker_num;
  int             j, k;
  double          patt_width     = 80.0; /* [mm] */
  double          patt_center[2] = {0.0, 0.0};
  double          patt_trans[3][4];

  arVideoCapStart();

  for (;;) {

    /* Grab a video frame */
    if ((dataPtr = (ARUint8 *)arVideoGetImage()) == NULL) {
      /* Retry soon. */
      arUtilSleep(2);
      continue;
    }

    if (count == 0) {
      arUtilTimerReset();
    }
    count++;
    
    if (!(count % 20)) {
      pthread_mutex_lock(&mutex);
      fps = (double)count/arUtilTimer();
      pthread_mutex_unlock(&mutex);
    }

    /* Detect the markers in the video frame */
    if (arDetectMarkerLite(dataPtr, thresh, &marker_info, &marker_num) < 0) {
      fprintf(stderr, "headtracker.c: arDetectMarkerLite() failed.\n");
      cleanup();
      exit(-1);
    }

    arVideoCapNext();
   
    /* Check for object visibility */
    k = -1;
    for (j = 0; j < marker_num; j++) {
      if (patt_id == marker_info[j].id) {
	if ((k == -1) || (marker_info[k].cf < marker_info[j].cf)) {
	  k = j;
	}
      }
    }
  
    /* Get the transformation between the marker and the real camera */
    if (k != -1) {
      double user_trans[3][4];
      double rot[4], pos[4];
      int i, j;

      arGetTransMatCont(&marker_info[k],
			patt_trans, patt_center, patt_width, patt_trans);
      /* patt_trans is the transformation from camera CS to pattern CS. */
      
      /* Use the marker position... */
      pthread_mutex_lock(&mutex);

      /* Compute trans patt -> user CS. */
      /* This is guess. :( */
      if (arUtilMatMul(origin_trans_inv, patt_trans, user_trans) < 0) {
        fprintf(stderr, "headtracker.c: arUtilMatMul() failed.\n");
      }

      if (0 != arUtilMat2QuatPos(user_trans, rot, pos)) {
	fprintf(stderr, "headtracker.c: arUtilMat2QuatPos() failed.\n");
      } else {
	/* Compute the new current position using exponential averaging. */
	current_pos[0] = freshness * pos[0] +
	  (1.0 - freshness) * current_pos[0];
	current_pos[1] = freshness * pos[1] +
	  (1.0 - freshness) * current_pos[1];
	current_pos[2] = freshness * pos[2] +
	  (1.0 - freshness) * current_pos[2];
	/* Compute the euler angles. */
        double heading, pitch, roll;

        if (rot[0]*rot[2] - rot[1]*rot[3] == 0.5) {
          heading = 2.0 * asin(rot[1]/cos(M_PI/4.0));
          pitch   = M_PI/2.0;
          roll    = 0.0;
        } else if (rot[0]*rot[2] - rot[1]*rot[3] == -0.5) {
          heading = 2.0 * asin(rot[1]/cos(M_PI/4.0));
          pitch   = -M_PI/2.0;
          roll    = 0.0;
        } else {
          heading = atan(2.0*(rot[0]*rot[1] + rot[2]*rot[3]) /
                         (rot[0]*rot[0] - rot[1]*rot[1] -
                          rot[2]*rot[2] + rot[3]*rot[3]));
          pitch   = asin(2.0*(rot[0]*rot[2] - rot[1]*rot[3]));
          roll    = atan(2.0*(rot[0]*rot[3] + rot[1]*rot[2]) /
                         (rot[0]*rot[0] + rot[1]*rot[1] -
                          rot[2]*rot[2] - rot[3]*rot[3]));
        }

	current_rot_deg[0] =
	  freshness * heading * (180.0 / M_PI) +
	  (1.0 - freshness) * current_rot_deg[0];
	current_rot_deg[1] =
	  freshness * -roll * (180.0 / M_PI) +
	  (1.0 - freshness) * current_rot_deg[1];
	current_rot_deg[2] =
	  freshness * -pitch * (180.0 / M_PI) +
	  (1.0 - freshness) * current_rot_deg[2];

	/* Special case: Initialialize origin position. */
	if (startup) {
	  startup = 0;
          if (arUtilMatInv(patt_trans, origin_trans_inv) < 0) {
            fprintf(stderr, "headtracker.c: arUtilMatInv() failed.\n");
          }
	}
      }
      pthread_mutex_unlock(&mutex);      
    } else {
      fprintf(stderr, "headtracker.c: No marker detected.\n");
    }
  }
  return NULL;
}

/*---------------------------------------------------------------------------*/
static void*  transmitter(void* arg)
{
  int outsocket;
  struct in_addr     destIP;
  struct timespec    delay;
  struct sockaddr_in destAddr;

  delay.tv_sec  = period / 1000;
  delay.tv_nsec = (period % 1000) * 1000000;

  if (-1 == (outsocket = socket(PF_INET, SOCK_DGRAM, 0))) {
    perror("headtracker.c: socket()");
    exit(-1);
  }
  
  if (0 == inet_aton(host, &destIP)) {
    fprintf(stderr, "headtracker.c: inet_aton() failed.\n");
    exit(-1);
  }

  fprintf(stderr, "headtracker.c: Created socket. Destination = %s:%d!\n",
	 host, port);
  

  bzero((char *)&destAddr, sizeof(destAddr));

  destAddr.sin_family      = PF_INET;
  destAddr.sin_addr.s_addr = destIP.s_addr;
  destAddr.sin_port        = htons(port);

  for (;;) {
    const int packetSize = 1 + 6*sizeof(float);
    char msg[packetSize];

    nanosleep(&delay, NULL);

    /* Read current tracker state. */
    pthread_mutex_lock(&mutex);

    msg[0] = 1;                                      // Version.
    encodeFloat(&msg[1], (float)current_rot_deg[2]); // heading-deg
    encodeFloat(&msg[5], (float)current_rot_deg[0]); // roll-deg
    encodeFloat(&msg[9], (float)current_rot_deg[1]); // pitch-deg
    encodeFloat(&msg[13],                            // pos-x (right)
		(float)(-current_pos[0]));
    encodeFloat(&msg[17],                            // pos-y (up) 
		(float)(current_pos[1]));
    encodeFloat(&msg[21],                            // pos-z (back)
		(float)(-current_pos[2]));

    /*printf("Sent: pos = (%f %f %f), rot = (%f %f %f)\n",
           current_pos[0],
           current_pos[1],
           current_pos[2],
           current_rot_deg[2],
           current_rot_deg[0],
           current_rot_deg[1]);*/
    /*
    printf("Received pos = (%f %f %f)\n",
           decodeFloat(&msg[13]),
           decodeFloat(&msg[17]),
           decodeFloat(&msg[21]));
    */

    pthread_mutex_unlock(&mutex);

    /* Send current tracker state. */
    if (-1 == sendto(outsocket, msg, sizeof(msg), 0,
		     (struct sockaddr*)&destAddr, sizeof(destAddr))) {
      perror("headtracker.c: sendto()");
      exit(-1);
    }
  }

  close(outsocket);
}

/*---------------------------------------------------------------------------*/
static void   encodeFloat(char* buf, float val)
{
  uint32_t tmp = htonl(*(uint32_t *)&val);
  memcpy(buf, &tmp, sizeof(uint32_t));
}

static float   decodeFloat(char* buf)
{
  uint32_t tmp = ntohl(*(uint32_t *)buf);
  return *(float *)&tmp;
}

static void   encodeFixed(char* buf, float val)
{
  int fixed = (int)(val * 65536.0f); 
  
  buf[0] = (unsigned char)(fixed >> 24); 
  buf[1] = (unsigned char)(fixed >> 16); 
  buf[2] = (unsigned char)(fixed >> 8); 
  buf[3] = (unsigned char)fixed; 
}

static float   decodeFixed(char* buf)
{
  int tmp =
    (((buf[0]) & 0xff) << 24) |
    (((buf[1]) & 0xff) << 16) |
    (((buf[2]) & 0xff) << 8) |
    ((buf[3]) & 0xff);
  return (float)tmp / 65536.0f;
}
